import requests
import json
from uuid import uuid4

SONARQUBE_URL = "<sonarqube_url>"
TOKEN = "<sonarqube_token>" # Replace with a user token from a SonarQube admin user who has access to all the projects
DESIRED_LANGUAGES = ["vhdl", "verilog", "tcl"]

def get_all_projects():
    projects = []
    page = 1
    while True:
        response = requests.get(
            f"{SONARQUBE_URL}/api/projects/search",
            auth=(TOKEN, ""),
            params={"ps": 500, "p": page},
            timeout=10
        )
        if response.status_code != 200:
            raise Exception(f"Failed to fetch projects: {response.status_code} - {response.text}")
        data = response.json()
        projects.extend(data["components"])
        if page * 500 >= data["paging"]["total"]:
            break
        page += 1
    return projects

def get_language_distribution(project_key):
    response = requests.get(
        f"{SONARQUBE_URL}/api/measures/component",
        auth=(TOKEN, ""),
        params={"component": project_key, "metricKeys": "ncloc_language_distribution"},
        timeout=10
    )
    if response.status_code != 200:
        return None
    data = response.json()
    measures = data.get("component", {}).get("measures", [])
    if not measures:
        return None
    # Parse ncloc_language_distribution (e.g., "java=1000;js=500")
    value = measures[0].get("value", "")
    if not value:
        return None
    language_loc = {}
    for lang_entry in value.split(";"):
        if "=" in lang_entry:
            lang, loc = lang_entry.split("=")
            language_loc[lang] = int(loc)
    return language_loc

def main():
    try:
        projects = get_all_projects()
    except Exception as e:
        print(f"Error fetching projects: {e}")
        return

    matching_projects = []
    for project in projects:
        project_key = project["key"]
        project_name = project["name"]
        language_loc = get_language_distribution(project_key)
        if not language_loc:
            continue
        relevant_langs = {lang: loc for lang, loc in language_loc.items() if lang in DESIRED_LANGUAGES}
        if relevant_langs:
            matching_projects.append({"key": project_key, "name": project_name, "languages": relevant_langs})

    if not matching_projects:
        print("No projects found analyzing the specified languages.")
        return

    for project in matching_projects:
        print(f"{project['name']} ({project['key']}):")
        for lang, loc in project["languages"].items():
            print(f"  - {lang}: {loc} lines of code")

if __name__ == "__main__":
    main()
