/*
 * Decompiled with CFR 0.152.
 */
package com.lintyservices.sonar.plugins.bugfinder;

import com.lintyservices.sonar.plugins.bugfinder.BugFinderException;
import com.lintyservices.sonar.plugins.bugfinder.BugFinderOutput;
import com.lintyservices.sonar.plugins.hdl.LintyPaths;
import com.lintyservices.utils.FileUtils;
import com.lintyservices.utils.LintyException;
import com.lintyservices.yosys.objects.YosysCDC;
import com.lintyservices.yosys.objects.YosysRDC;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.lang.invoke.CallSite;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Properties;
import java.util.Set;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.sonar.api.batch.sensor.SensorContext;
import org.sonar.api.config.Configuration;

public class BugFinder {
    public static final String LOG_PREFIX = "[BugFinder] ";
    private static final Logger LOG = LoggerFactory.getLogger(BugFinder.class);
    private static final String DESIGN_READ_AND_HIERARCHY = "read_hierarchy";
    private static final String DESIGN_PROC_AND_HLP_OPT = "proc_and_hlp_opt";
    private static final String DESIGN_PREP_AND_OPT = "prep_and_opt";
    private static final String DESIGN_PROC_AND_HLP_OPT_AND_FLATTEN = "proc_and_hlp_opt_and_flatten";
    private static final String DESIGN_PROC_AND_FLATTEN_AND_OPT = "proc_and_flatten_and_opt";
    private static final String DESIGN_PROC_AND_HLP_OPT_AND_ALL_MODULE_FLATTEN = "proc_and_hlp_opt_and_all_module_flatten";
    private static final String DESIGN_CLOCK_DOMAINS = "clock_domains";
    private static final String STEP_LICENSE = "license";
    private static final String STEP_PLUGINS = "plugins";
    private static final String STEP_READ = "read";
    private static final String STEP_SYNTHESIZED_OBJECTS = "synthesized_objects";
    private static final String STEP_HIERARCHY = "hierarchy";
    private static final String STEP_METRICS = "metrics";
    private static final String STEP_RULES = "rules";
    private static final String PROC_AND_HLP_OPT = "yosys proc -noopt\nyosys hlp_opt\n";
    private static final String PROC_AND_OPT = "yosys proc \nyosys opt\n";
    private static final String PROC = "yosys proc -noopt\n";
    private static final String OPT = "yosys opt\n";
    private static final String FLATTEN = "yosys flatten -scopename\n";
    private static final String FLATTEN_NO_CLEANUP = "yosys flatten -scopename -nocleanup\n";
    private final Configuration config;
    private final File bugfinderWorkDir;
    private final String topModule;
    private final Path logsDirPath;
    private final Path rtlilDirPath;
    private final Path logOutputFilePath;
    private final Path designInformationDate;
    private final Path logStepsFilePath;
    private final Path mainTclBuildScriptPath;
    private final Path graphTclBuildScriptPath;
    private final Path nonSynthesizedObjectsTclBuildScriptPath;
    private final LintyPaths lintyPaths;
    private Process process;
    private BufferedWriter stdinWriter;
    private Integer numberOfTopLevelInputRegistrations;
    private Integer numberOfLowLevelInputRegistrations;

    public BugFinder(SensorContext context) {
        this.config = context.config();
        this.lintyPaths = new LintyPaths(this.config);
        this.bugfinderWorkDir = this.lintyPaths.bugfinderWorkDir();
        if (!this.config.get("sonar.hdl.topModule").isPresent()) {
            throw new LintyException("[BUGFINDER] sonar.hdl.topModule property is not set");
        }
        this.topModule = (String)this.config.get("sonar.hdl.topModule").get();
        this.logsDirPath = Paths.get(String.valueOf(this.bugfinderWorkDir) + File.separator + "logs", new String[0]);
        this.rtlilDirPath = Paths.get(String.valueOf(this.bugfinderWorkDir) + File.separator + "rtlil", new String[0]);
        this.logOutputFilePath = Paths.get(String.valueOf(this.logsDirPath) + File.separator + "output.log", new String[0]);
        this.designInformationDate = Paths.get(String.valueOf(this.bugfinderWorkDir) + File.separator + "design_information_date.txt", new String[0]);
        this.logStepsFilePath = Paths.get(String.valueOf(this.logsDirPath) + File.separator + "steps.log", new String[0]);
        this.mainTclBuildScriptPath = Paths.get(String.valueOf(this.bugfinderWorkDir) + File.separator + "build.tcl", new String[0]);
        this.graphTclBuildScriptPath = Paths.get(String.valueOf(this.bugfinderWorkDir) + File.separator + "graphs.tcl", new String[0]);
        this.nonSynthesizedObjectsTclBuildScriptPath = Paths.get(String.valueOf(this.bugfinderWorkDir) + File.separator + "build_non_synthesized_objects.tcl", new String[0]);
    }

    public void setNumberOfTopLevelInputRegistrations(int numberOfRegistrations) {
        this.numberOfTopLevelInputRegistrations = numberOfRegistrations;
    }

    public void setNumberOfLowLevelInputRegistrations(int numberOfRegistrations) {
        this.numberOfLowLevelInputRegistrations = numberOfRegistrations;
    }

    public BugFinderOutput execute() throws BugFinderException {
        try {
            this.createWorkDir();
            this.createRtlilDir();
            this.createLogFiles();
            this.createNonSynthesizedObjectsTclBuildScript();
            this.start("Running Yosys to detect non-synthesized objects");
            this.runCommands("tcl " + String.valueOf(this.nonSynthesizedObjectsTclBuildScriptPath));
            this.stop(Set.of(STEP_LICENSE, STEP_PLUGINS, STEP_READ, STEP_SYNTHESIZED_OBJECTS));
            this.createMainTclBuildScript();
            this.start("Running Yosys to detect issues");
            this.runCommands("tcl " + String.valueOf(this.mainTclBuildScriptPath));
            this.stop(Set.of(STEP_HIERARCHY, STEP_METRICS));
            BugFinderOutput bugFinderOutput = new BugFinderOutput(this.config);
            this.createGraphTclBuildScript(bugFinderOutput);
            this.start("Running Yosys to generate graphs");
            this.runCommands("tcl " + String.valueOf(this.graphTclBuildScriptPath));
            this.stop(Set.of(STEP_RULES));
            this.writeDesignInformationDateFile();
            return bugFinderOutput;
        }
        catch (IOException | InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new BugFinderException(e);
        }
    }

    private void checkYosysOutput(Set<String> stepsToCheck) {
        try (FileInputStream input = new FileInputStream(this.logStepsFilePath.toFile());){
            Properties properties = new Properties();
            properties.load(input);
            String requestSupport = "Or request support by sending an email to support@linty-services.com\n";
            String checkYosysOutput = "Check Yosys output: " + FileUtils.sanitizedPath(this.logOutputFilePath.toAbsolutePath().toString()) + "\n";
            if (stepsToCheck.contains(STEP_LICENSE) && !properties.containsKey(STEP_LICENSE)) {
                throw new LintyException("\n\n[BugFinder] Your Tabby CAD (Linty Services Edition) license is invalid.\nCheck that the license is properly passed to the Linty Scanner. See https://doc.linty-services.com/doc/scan.html#run-linty-scan\nOr request a license. See https://doc.linty-services.com/doc/configure.html#set-linty-license-keys\n" + requestSupport);
            }
            if (stepsToCheck.contains(STEP_PLUGINS) && !properties.containsKey(STEP_PLUGINS)) {
                throw new LintyException("\n\n[BugFinder] Linty Yosys plugins cannot be loaded.\nCheck that your Linty Scanner is compatible with your Linty Server. See https://doc.linty-services.com/doc/scan.html#compatibility-matrix-with-linty-server\n" + requestSupport);
            }
            if (stepsToCheck.contains(STEP_READ) && !properties.containsKey(STEP_READ)) {
                throw new LintyException("\n\n[BugFinder] Your VHDL/Verilog files cannot be read.\nCheck Yosys build script: " + FileUtils.sanitizedPath(this.lintyPaths.yosysReadFile().getAbsolutePath()) + "\n" + checkYosysOutput + "See https://doc.linty-services.com/doc/scan_bugfinder.html\n" + requestSupport);
            }
            if (stepsToCheck.contains(STEP_SYNTHESIZED_OBJECTS) && !properties.containsKey(STEP_SYNTHESIZED_OBJECTS)) {
                throw new LintyException("\n\n[BugFinder] Your IP cannot be elaborated.\nCheck Yosys build script: " + FileUtils.sanitizedPath(this.lintyPaths.yosysReadFile().getAbsolutePath()) + "\n" + checkYosysOutput + "See https://doc.linty-services.com/doc/scan_bugfinder.html\n" + requestSupport);
            }
            if (stepsToCheck.contains(STEP_HIERARCHY) && !properties.containsKey(STEP_HIERARCHY)) {
                throw new LintyException("\n\n[BugFinder] Hierarchy of your IP cannot be built.\nCheck Yosys build script: " + FileUtils.sanitizedPath(this.lintyPaths.yosysHierarchyFile().getAbsolutePath()) + "\n" + checkYosysOutput + "See https://doc.linty-services.com/doc/scan_bugfinder.html\n" + requestSupport);
            }
            if (stepsToCheck.contains(STEP_METRICS) && !properties.containsKey(STEP_METRICS)) {
                throw new LintyException("\n\n[BugFinder] Your IP metrics cannot be computed.\n" + checkYosysOutput + requestSupport);
            }
            if (stepsToCheck.contains(STEP_RULES) && !properties.containsKey(STEP_RULES)) {
                throw new LintyException("\n\n[BugFinder] BugFinder rules cannot be checked.\n" + checkYosysOutput + requestSupport);
            }
        }
        catch (IOException ex) {
            throw new LintyException("Cannot read " + String.valueOf(this.logStepsFilePath) + " file");
        }
    }

    private void createWorkDir() throws IOException {
        Files.createDirectories(this.bugfinderWorkDir.toPath(), new FileAttribute[0]);
    }

    private void createLogFiles() throws IOException {
        Files.createDirectories(this.logsDirPath, new FileAttribute[0]);
        Files.createFile(this.logOutputFilePath, new FileAttribute[0]);
        Files.createFile(this.logStepsFilePath, new FileAttribute[0]);
    }

    private void createRtlilDir() throws IOException {
        Files.createDirectory(this.rtlilDirPath, new FileAttribute[0]);
    }

    private StringBuilder tclScriptInit(Set<String> plugins) {
        StringBuilder text = new StringBuilder().append(this.logInfo("Init Yosys and Verific")).append("yosys logger -time\n").append("yosys -import\n").append(this.logToSteps(STEP_LICENSE)).append("source $::env(TABBY_CAD_HOME)/lib/tcl8.6/init.tcl\n").append(this.yosysCommand("verific -cfg db_cse 0")).append(this.yosysCommand("verific -cfg db_infer_wide_muxes_post_elaboration 0")).append(this.yosysCommand("verific -set-warning VHDL-1241")).append(this.yosysCommand("verific -set-warning VHDL-1045")).append(this.yosysCommand("verific -set-warning VERI-1906")).append(this.yosysCommand("setattr -set keep 1 w:\\*")).append("\n\n").append(this.logInfo("Load Yosys plugins"));
        for (String plugin : plugins) {
            text.append(this.yosysCommandWithoutNewLine("plugin -i ")).append(plugin).append("\n");
        }
        text.append(this.yosysCommand("plugin -l")).append(this.logToSteps(STEP_PLUGINS)).append("\n\n").append(this.logInfo("Read VHDL Files")).append(this.yosysCommandWithoutNewLine("hlp_read -read ")).append(this.lintyPaths.yosysReadFile().getAbsolutePath()).append("\n").append(this.logToSteps(STEP_READ)).append("\n\n");
        return text;
    }

    private void createNonSynthesizedObjectsTclBuildScript() throws IOException {
        LOG.info("{}Creating Non-synthesized-objects TCL build script: {}", (Object)LOG_PREFIX, (Object)FileUtils.sanitizedPath(this.nonSynthesizedObjectsTclBuildScriptPath.toAbsolutePath().toString()));
        Set<String> plugins = Set.of("hlp_read", "hlp_synthesized_objects");
        StringBuilder content = this.tclScriptInit(plugins).append(this.logInfo("Get Synthesized Objects"));
        Object synthesizedObjectsParams = this.retrieveHierarchyParameters();
        synthesizedObjectsParams = ((String)synthesizedObjectsParams).isEmpty() ? "" : " -params " + (String)synthesizedObjectsParams;
        content.append(this.callPass("hlp_synthesized_objects", "-top " + this.topModule + (String)synthesizedObjectsParams)).append(this.logToSteps(STEP_SYNTHESIZED_OBJECTS)).append("\n");
        Files.writeString(this.nonSynthesizedObjectsTclBuildScriptPath, (CharSequence)content, new OpenOption[0]);
    }

    private void createMainTclBuildScript() throws IOException {
        LOG.info("{}Creating Main TCL build script: {}", (Object)LOG_PREFIX, (Object)FileUtils.sanitizedPath(this.mainTclBuildScriptPath.toAbsolutePath().toString()));
        Set<String> plugins = Set.of("hlp_constant_output_ports", "hlp_clock_domains", "hlp_clocks", "hlp_combinational_loops", "hlp_design_hierarchy", "hlp_directly_connected_ports", "hlp_enables", "hlp_ff_no_reset", "hlp_fsm", "hlp_hierarchy", "hlp_libraries", "hlp_modules", "hlp_opt", "hlp_read", "hlp_resets", "hlp_unconnected_ports", "hlp_unregistered_output_ports", "hlp_unregistered_input_ports");
        StringBuilder content = this.tclScriptInit(plugins).append(this.logInfo("Build Hierarchy")).append(this.yosysCommandWithoutNewLine("hlp_hierarchy -hierarchy ")).append(this.lintyPaths.yosysHierarchyFile().getAbsolutePath()).append("\n").append(this.logToSteps(STEP_HIERARCHY)).append("\n\n").append(this.logInfo("Check rules")).append(this.aggregateTclRuleScripts()).append("\n\n").append(this.logInfo("Compute Metrics")).append("yosys logger -notime\n").append(this.executeRule("tee -q -o " + String.valueOf(this.bugfinderWorkDir) + File.separator + "custom_metrics.json stat -json\n", DESIGN_PROC_AND_HLP_OPT)).append("yosys logger -time\n").append(this.logToSteps(STEP_METRICS));
        Files.writeString(this.mainTclBuildScriptPath, (CharSequence)content, new OpenOption[0]);
    }

    private void createGraphTclBuildScript(BugFinderOutput bugFinderOutput) throws IOException {
        LOG.info("{}Creating Graph TCL build script: {}", (Object)LOG_PREFIX, (Object)FileUtils.sanitizedPath(this.graphTclBuildScriptPath.toAbsolutePath().toString()));
        Path cdcsDir = Paths.get(String.valueOf(this.bugfinderWorkDir) + File.separator + "cdcs", new String[0]);
        Files.createDirectory(cdcsDir, new FileAttribute[0]);
        Path rdcsDir = Paths.get(String.valueOf(this.bugfinderWorkDir) + File.separator + "rdcs", new String[0]);
        Files.createDirectory(rdcsDir, new FileAttribute[0]);
        StringBuilder content = new StringBuilder();
        content.append("yosys read_rtlil ").append(Paths.get(String.valueOf(this.rtlilDirPath) + File.separator + "clock_domains.rtlil", new String[0])).append("\n\n");
        HashMap cdcsAsMap = new HashMap();
        for (YosysCDC cdc : bugFinderOutput.cdcs()) {
            String key = cdc.fullOriginClockDomain() + ";" + cdc.fullTargetClockDomain();
            if (cdcsAsMap.containsKey(key)) {
                ((List)cdcsAsMap.get(key)).add(cdc);
                continue;
            }
            ArrayList<YosysCDC> cdcList = new ArrayList<YosysCDC>();
            cdcList.add(cdc);
            cdcsAsMap.put((CallSite)((Object)key), cdcList);
        }
        cdcsAsMap.keySet().stream().sorted().forEach(k -> {
            content.append("yosys select -clear\n").append("yosys select -none\n").append("yosys select -set all %\n");
            ((List)cdcsAsMap.get(k)).forEach(v -> content.append(String.format("  yosys select -clear\n  yosys select -set cdc_in c:%s %%ci %%cie*\n  yosys select -set cdc_out c:%s %%co %%coe*\n  yosys select -set cdc @cdc_in @cdc_out %%i c:%s c:%s\n  yosys select @cdc @cdc c:* %%i %%x %%u\n  yosys select -set cdc %%\n  yosys json -noscopeinfo -o %s/cdc_%s.json @cdc\n  yosys select @all @cdc %%u\n  yosys select -set all %%\n", this.protectString(v.target().cell().name()), this.protectString(v.origin().cell().name()), this.protectString(v.target().cell().name()), this.protectString(v.origin().cell().name()), cdcsDir, v.id())));
            content.append("\n").append(String.format("yosys json -noscopeinfo -o \"%s/cdcs_%s.json\" @all\n", cdcsDir, k));
        });
        HashMap rdcsAsMap = new HashMap();
        for (YosysRDC rdc : bugFinderOutput.rdcs()) {
            String key = rdc.fullOriginClockDomain() + ";" + rdc.fullTargetClockDomain();
            if (rdcsAsMap.containsKey(key)) {
                ((List)rdcsAsMap.get(key)).add(rdc);
                continue;
            }
            ArrayList<YosysRDC> rdcList = new ArrayList<YosysRDC>();
            rdcList.add(rdc);
            rdcsAsMap.put((CallSite)((Object)key), rdcList);
        }
        rdcsAsMap.keySet().stream().sorted().forEach(k -> {
            content.append("yosys select -clear\n").append("yosys select -none\n").append("yosys select -set all %\n");
            ((List)rdcsAsMap.get(k)).forEach(v -> content.append(String.format("  yosys select -clear\n  yosys select -set rdc_in c:%s %%ci %%cie*\n  yosys select -set rdc_out c:%s %%co %%coe*\n  yosys select -set rdc @rdc_in @rdc_out %%i c:%s c:%s\n  yosys select @rdc @rdc c:* %%i %%x %%u\n  yosys select -set rdc %%\n  yosys json -noscopeinfo -o %s/rdc_%s.json @rdc\n  yosys select @all @rdc %%u\n  yosys select -set all %%\n", this.protectString(v.target().cell().name()), this.protectString(v.origin().cell().name()), this.protectString(v.target().cell().name()), this.protectString(v.origin().cell().name()), rdcsDir, v.id())));
            content.append("\n").append(String.format("yosys json -noscopeinfo -o \"%s/rdcs_%s.json\" @all\n", rdcsDir, k));
        });
        Files.writeString(this.graphTclBuildScriptPath, (CharSequence)content, new OpenOption[0]);
    }

    private String aggregateTclRuleScripts() throws IOException {
        StringBuilder script = new StringBuilder();
        script.append(this.logInfo("Save designs")).append(this.saveYosysDesign(DESIGN_READ_AND_HIERARCHY)).append(this.dumpRtlil(Paths.get(String.valueOf(this.rtlilDirPath) + File.separator + "read_hierarchy.rtlil", new String[0]))).append(PROC_AND_HLP_OPT).append(this.saveYosysDesign(DESIGN_PROC_AND_HLP_OPT)).append(this.dumpRtlil(Paths.get(String.valueOf(this.rtlilDirPath) + File.separator + "read_hierarchy_proc_hlp_opt.rtlil", new String[0]))).append(this.loadYosysDesign(DESIGN_READ_AND_HIERARCHY)).append(PROC_AND_OPT).append(this.saveYosysDesign(DESIGN_PREP_AND_OPT)).append(this.dumpRtlil(Paths.get(String.valueOf(this.rtlilDirPath) + File.separator + "read_hierarchy_prep_and_opt.rtlil", new String[0]))).append(this.loadYosysDesign(DESIGN_PROC_AND_HLP_OPT)).append(FLATTEN).append(this.saveYosysDesign(DESIGN_PROC_AND_HLP_OPT_AND_FLATTEN)).append(this.dumpRtlil(Paths.get(String.valueOf(this.rtlilDirPath) + File.separator + "read_hierarchy_proc_hlp_opt_flatten.rtlil", new String[0]))).append(this.loadYosysDesign(DESIGN_READ_AND_HIERARCHY)).append(PROC).append(FLATTEN).append(OPT).append(this.saveYosysDesign(DESIGN_PROC_AND_FLATTEN_AND_OPT)).append(this.dumpRtlil(Paths.get(String.valueOf(this.rtlilDirPath) + File.separator + "read_hierarchy_proc_flatten_opt.rtlil", new String[0]))).append(this.loadYosysDesign(DESIGN_PROC_AND_HLP_OPT)).append(FLATTEN_NO_CLEANUP).append(this.saveYosysDesign(DESIGN_PROC_AND_HLP_OPT_AND_ALL_MODULE_FLATTEN)).append(this.dumpRtlil(Paths.get(String.valueOf(this.rtlilDirPath) + File.separator + "read_hierarchy_proc_hlp_opt_flatten_by_module.rtlil", new String[0]))).append("\n");
        script.append(this.logInfo("Generate design logic circuit graphs")).append(this.executeRule("yosys write_json -noscopeinfo " + String.valueOf(this.bugfinderWorkDir) + "/design.json\n", DESIGN_PREP_AND_OPT)).append(this.logInfo("Design Hierarchy")).append(this.executeRule(this.callPass("hlp_design_hierarchy"), DESIGN_READ_AND_HIERARCHY)).append(this.logInfo("List Modules")).append(this.executeRule(this.callPass("hlp_modules"), DESIGN_READ_AND_HIERARCHY)).append(this.logInfo("List Libraries")).append(this.executeRule(this.callPass("hlp_libraries"), DESIGN_READ_AND_HIERARCHY)).append(this.logInfo("BugFinderUnregisteredOutputPortsAwareVisitor")).append(this.executeRule(this.callPass("hlp_unregistered_output_ports"), DESIGN_PROC_AND_HLP_OPT_AND_ALL_MODULE_FLATTEN));
        if (this.numberOfTopLevelInputRegistrations != null) {
            script.append(this.logInfo("BugFinderUnregisteredTopLevelInputPortsAwareVisitor")).append(this.executeRule("yosys hlp_unregistered_input_ports -checkTopModule -levels " + this.numberOfTopLevelInputRegistrations + " -json -outputdir " + String.valueOf(this.bugfinderWorkDir) + " -outputFile unregistered_top_level_input_ports\n", DESIGN_PROC_AND_HLP_OPT));
        }
        if (this.numberOfLowLevelInputRegistrations != null) {
            script.append(this.logInfo("BugFinderUnregisteredLowLevelInputPortsAwareVisitor")).append(this.executeRule("yosys hlp_unregistered_input_ports -checkNonTopModules -levels " + this.numberOfLowLevelInputRegistrations + " -json -outputdir " + String.valueOf(this.bugfinderWorkDir) + " -outputFile unregistered_low_level_input_ports\n", DESIGN_PROC_AND_HLP_OPT));
        }
        script.append(this.logInfo("BugFinderUnconnectedPortsAwareVisitor")).append(this.executeRule(this.callPass("hlp_unconnected_ports"), DESIGN_PROC_AND_HLP_OPT_AND_ALL_MODULE_FLATTEN)).append(this.logInfo("BugFinderDirectlyConnectedPortsAwareVisitor")).append(this.executeRule(this.callPass("hlp_directly_connected_ports"), DESIGN_PROC_AND_HLP_OPT_AND_ALL_MODULE_FLATTEN)).append(this.logInfo("BugFinderConstantOutputPortsAwareVisitor")).append(this.executeRule(this.callPass("hlp_constant_output_ports"), DESIGN_PROC_AND_HLP_OPT_AND_ALL_MODULE_FLATTEN)).append(this.logInfo("BugFinderClocksAwareVisitor")).append(this.executeRule(this.callPass("hlp_clocks"), DESIGN_PROC_AND_HLP_OPT)).append(this.logInfo("BugFinderResetsAwareVisitor")).append(this.executeRule(this.callPass("hlp_resets"), DESIGN_PROC_AND_HLP_OPT)).append(this.logInfo("BugFinderEnablesAwareVisitor")).append(this.executeRule(this.callPass("hlp_enables"), DESIGN_PROC_AND_HLP_OPT)).append(this.logInfo("BugFinderFlipFlopNoResetAwareVisitor")).append(this.executeRule(this.callPass("hlp_ff_no_reset"), DESIGN_PROC_AND_HLP_OPT)).append(this.logInfo("BugFinderFsmsAwareVisitor")).append(this.executeRule(this.yosysCommand("proc") + this.yosysCommand("bmuxmap -pmux") + this.yosysCommand("setattr -set fsm_encoding {\"auto\"} w:\\* x:* %d a:fsm_encoding %d a:enum_value_* %i") + this.yosysCommand("opt -nodffe -nosdff") + this.yosysCommand("fsm_detect -ignore-self-reset") + this.yosysCommand("fsm_extract") + this.callPass("hlp_fsm"), DESIGN_READ_AND_HIERARCHY)).append(this.dumpRtlil(Paths.get(String.valueOf(this.rtlilDirPath) + File.separator + "fsm.rtlil", new String[0])));
        Path loopsDirectory = Paths.get(String.valueOf(this.bugfinderWorkDir) + File.separator + "combinational_loops", new String[0]);
        Files.createDirectory(loopsDirectory, new FileAttribute[0]);
        script.append(this.logInfo("BugFinderCombinationalLoopsAwareVisitor")).append(this.executeRule(this.callPass("hlp_combinational_loops"), DESIGN_PROC_AND_FLATTEN_AND_OPT)).append(String.format("set f [open \"%s/combinational_loops.csv\" r]\nset number_of_loops [lindex [split [read $f] \"\\n\"] 0]\nfor {set i 1} {$i <= $number_of_loops} {incr i} {\n    yosys select -set loop a:combinational_loop_id=*|$i|*\n    yosys select @loop @loop c:* %%i %%x %%u\n    yosys json -noscopeinfo -o %s/combinational_loop_$i.json\n}\nclose $f\n", this.bugfinderWorkDir, loopsDirectory)).append(this.logInfo("BugFinderClockDomainsVisitor")).append(this.executeRule(this.callPass("hlp_clock_domains"), DESIGN_PROC_AND_FLATTEN_AND_OPT)).append(this.saveYosysDesign(DESIGN_CLOCK_DOMAINS)).append(this.dumpRtlil(Paths.get(String.valueOf(this.rtlilDirPath) + File.separator + "clock_domains.rtlil", new String[0]))).append(this.logToSteps(STEP_RULES));
        return script.toString();
    }

    private void start(String label) throws IOException {
        LOG.info(LOG_PREFIX + label);
        ProcessBuilder pb = new ProcessBuilder("bash", "-c", "yosys").redirectOutput(ProcessBuilder.Redirect.appendTo(this.logOutputFilePath.toFile())).redirectError(ProcessBuilder.Redirect.appendTo(this.logOutputFilePath.toFile()));
        pb.directory(this.lintyPaths.baseDir());
        this.process = pb.start();
        this.stdinWriter = new BufferedWriter(new OutputStreamWriter(this.process.getOutputStream()));
    }

    private void stop(Set<String> stepsToCheck) throws InterruptedException, IOException {
        this.runCommands("exit");
        this.stdinWriter.flush();
        this.stdinWriter.close();
        this.process.waitFor();
        this.process.destroy();
        this.sanitizeLog();
        this.checkYosysOutput(stepsToCheck);
    }

    private void sanitizeLog() {
        String content;
        try {
            content = Files.readString(this.logOutputFilePath, StandardCharsets.UTF_8);
        }
        catch (IOException e) {
            throw new LintyException("Cannot read file " + FileUtils.sanitizedPath(this.logOutputFilePath.toAbsolutePath().toString()), e);
        }
        content = content.replace("/usr/src/", "./");
        try {
            Files.writeString(this.logOutputFilePath, (CharSequence)content, StandardCharsets.UTF_8, new OpenOption[0]);
        }
        catch (IOException e) {
            throw new LintyException("Cannot write file " + FileUtils.sanitizedPath(this.logOutputFilePath.toAbsolutePath().toString()), e);
        }
    }

    private void writeDesignInformationDateFile() {
        LocalDateTime now = LocalDateTime.now();
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
        try {
            Files.writeString(this.designInformationDate, (CharSequence)now.format(formatter), StandardCharsets.UTF_8, new OpenOption[0]);
        }
        catch (IOException e) {
            throw new LintyException("Cannot write file " + FileUtils.sanitizedPath(this.designInformationDate.toAbsolutePath().toString()), e);
        }
    }

    private void runCommands(String command) throws IOException {
        this.stdinWriter.write(command + "\n");
        this.stdinWriter.flush();
    }

    private String yosysCommand(String command) {
        return this.yosysCommandWithoutNewLine(command) + "\n";
    }

    private String yosysCommandWithoutNewLine(String command) {
        return "yosys " + command;
    }

    private String logToSteps(String step) {
        return this.yosysCommand("exec -- echo \"" + step + ": passed\" >> " + String.valueOf(this.logStepsFilePath));
    }

    private String saveYosysDesign(String name) {
        return this.yosysCommand("design -save " + name);
    }

    private String dumpRtlil(Path outputPath) {
        return this.yosysCommand("dump -o " + String.valueOf(outputPath.toAbsolutePath()));
    }

    private String loadYosysDesign(String name) {
        return this.yosysCommand("design -load " + name);
    }

    private String callPass(String pass) {
        return this.callPass(pass, "");
    }

    private String callPass(String pass, String parameters) {
        return "yosys " + pass + " " + parameters + " -json -outputdir " + String.valueOf(this.bugfinderWorkDir) + "\n";
    }

    private String executeRule(String rule, String design) {
        return this.logInfo(rule.substring(0, rule.length() - 1)) + this.loadYosysDesign(design) + rule + "\n";
    }

    private String logInfo(String info) {
        return "puts \"" + info.replace("\"", "\\\"") + "\"\n";
    }

    private String retrieveHierarchyParameters() {
        ArrayList<String> params = new ArrayList<String>();
        try (Stream<String> lines = Files.lines(this.lintyPaths.yosysHierarchyFile().toPath());){
            for (String line : lines.toList()) {
                if (!line.trim().startsWith(STEP_HIERARCHY)) continue;
                String[] chparam = line.trim().split("-chparam");
                for (int i = 1; i < chparam.length; ++i) {
                    String[] chparamSplit = chparam[i].split("\\s+");
                    params.add(chparamSplit[1]);
                    params.add(chparamSplit[2]);
                }
                break;
            }
        }
        catch (IOException e) {
            throw new LintyException("Cannot read file: " + FileUtils.sanitizedPath(this.lintyPaths.yosysHierarchyFile().getAbsolutePath()), e);
        }
        return String.join((CharSequence)":", params);
    }

    private String protectString(String s) {
        return s.replaceAll("[\\\\$]", "\\\\$0");
    }
}

